﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/gamelift/GameLift_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GameLift {
namespace Model {

/**
 * <p>This key-value pair can store custom data about a game session. For example,
 * you might use a <code>GameProperty</code> to track a game session's map, level
 * of difficulty, or remaining time. The difficulty level could be specified like
 * this: <code>{"Key": "difficulty", "Value":"Novice"}</code>. </p> <p> You can set
 * game properties when creating a game session. You can also modify game
 * properties of an active game session. When searching for game sessions, you can
 * filter on game property keys and values. You can't delete game properties from a
 * game session. </p> <p>For examples of working with game properties, see <a
 * href="https://docs.aws.amazon.com/gamelift/latest/developerguide/gamelift-sdk-client-api.html#game-properties">Create
 * a game session with properties</a>. </p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/gamelift-2015-10-01/GameProperty">AWS
 * API Reference</a></p>
 */
class GameProperty {
 public:
  AWS_GAMELIFT_API GameProperty() = default;
  AWS_GAMELIFT_API GameProperty(Aws::Utils::Json::JsonView jsonValue);
  AWS_GAMELIFT_API GameProperty& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GAMELIFT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The game property identifier.</p>  <p>Avoid using periods (".") in
   * property keys if you plan to search for game sessions by properties. Property
   * keys containing periods cannot be searched and will be filtered out from search
   * results due to search index limitations.</p>
   */
  inline const Aws::String& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::String>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::String>
  GameProperty& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The game property value.</p>
   */
  inline const Aws::String& GetValue() const { return m_value; }
  inline bool ValueHasBeenSet() const { return m_valueHasBeenSet; }
  template <typename ValueT = Aws::String>
  void SetValue(ValueT&& value) {
    m_valueHasBeenSet = true;
    m_value = std::forward<ValueT>(value);
  }
  template <typename ValueT = Aws::String>
  GameProperty& WithValue(ValueT&& value) {
    SetValue(std::forward<ValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_key;

  Aws::String m_value;
  bool m_keyHasBeenSet = false;
  bool m_valueHasBeenSet = false;
};

}  // namespace Model
}  // namespace GameLift
}  // namespace Aws
