// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A0__F356

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_256_0_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x2caa5c71u, 0xc7705462u, 0, 138 }, // 2caa5c71c7705462 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__16_16__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x1f4f204du, 0x23e53e6eu, 0, 63 }, // 1f4f204d23e53e6e = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x6bcb518cu, 0xc85ecc8bu, 0, 80 }, // 6bcb518cc85ecc8b = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__16_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x034cdb11u, 0xca50b755u, 0, 155 }, // 034cdb11ca50b755 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0x9bc81a13u, 0x7ee75f52u, 40, 46 }, // 9bc81a137ee75f52 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x5c46e00cu, 0xceb82738u, 40, 63 }, // 5c46e00cceb82738 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xe2de9a69u, 0x5049e241u, 40, 80 }, // e2de9a695049e241 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x4f7371b5u, 0xe4539171u, 97, 138 }, // 4f7371b5e4539171 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__32_32__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x1c5d0402u, 0xd0e4b7b8u, 120, 80 }, // 1c5d0402d0e4b7b8 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x32b9750au, 0xd9520c38u, 126, 46 }, // 32b9750ad9520c38 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xd16ff3e4u, 0x86d1f49du, 126, 138 }, // d16ff3e486d1f49d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__64_32__CO__wave1_warp4_stg1--Arch_gfx950
    { 0x803bb1a1u, 0x132e1732u, 132, 138 }, // 803bb1a1132e1732 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊bf16@16_256_0_T_F_0___gfx950__P__64_64__CO__wave1_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{ 6, 6, 7, 7, 0, 0, 0, 0, 0, 0},
 { 2, 3, 3, 0, 0, 0, 7, 7, 7, 7},
 { 3, 3, 2, 6, 6, 6, 7, 7, 7, 7},
 { 2, 2, 1, 5, 5, 5,10,11,11,11},
 { 4, 5, 5, 5, 4, 4, 5, 5, 9, 5},
 { 1, 1, 4, 8, 5, 5, 5, 5, 5, 5},
 { 1, 5, 5, 5, 5, 5, 8, 5, 5,11},
 { 1, 5, 5, 5, 5, 5, 8, 5, 5, 5},
 { 1, 5, 5, 5, 5, 5, 5, 5, 5, 5},
 { 5, 5, 5, 5, 5, 4, 4, 5, 5, 5}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A0__F356 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 2;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 2;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 256
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 256 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

