// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dq.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dq__A0__F116

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_160_0_T_F_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xb3f1e5afu, 0x6422b44du, 0, 63 }, // b3f1e5af6422b44d = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x5b4ed57au, 0x7de2899eu, 0, 80 }, // 5b4ed57a7de2899e = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__16_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x32452ac4u, 0x9d04addfu, 0, 6 }, // 32452ac49d04addf = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x51ea0c51u, 0xf3ec4352u, 0, 155 }, // 51ea0c51f3ec4352 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__16_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xeddf5840u, 0x283457adu, 40, 63 }, // eddf5840283457ad = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xa2edb8fbu, 0x17eec6e8u, 40, 80 }, // a2edb8fb17eec6e8 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__32_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x42c192ceu, 0x20af5cd0u, 97, 46 }, // 42c192ce20af5cd0 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__32_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0xfe02f3f4u, 0xfc514c50u, 97, 138 }, // fe02f3f4fc514c50 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__32_32__CO__wave1_warp4_stg1--Arch_gfx950
    { 0xfa7e60c2u, 0x67c32131u, 120, 155 }, // fa7e60c267c32131 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__64_16__CO__wave3_warp4_stg1--Arch_gfx950
    { 0xf61fb303u, 0x2b4bfae0u, 126, 46 }, // f61fb3032b4bfae0 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x92210b12u, 0x6135d551u, 132, 46 }, // 92210b126135d551 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__64_64__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x72b58148u, 0x8614139fu, 132, 138 }, // 72b581488614139f = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_dq/FONLY__＊fp16@16_160_0_T_F_0___gfx950__P__64_64__CO__wave1_warp4_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dq_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dq_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx950_mod0
{{ 5, 7, 7,11,11,11,11,11,11,11},
 { 3, 1, 3, 1, 3, 3, 3, 6, 6, 6},
 { 3, 1, 1, 3, 6, 6, 6, 6, 6, 6},
 { 1, 1, 0, 0, 6, 6, 6, 6, 6, 6},
 { 2, 2, 4, 4, 4, 9, 9, 9,10,10},
 { 2, 8, 8, 8, 8, 8, 8, 8, 8, 8},
 { 2, 8, 8, 8, 8, 9, 9, 9,10, 6},
 { 2, 8, 8, 8, 8, 9, 9,10,10,10},
 { 2, 8, 8, 8, 8, 9, 9, 8, 8, 8},
 { 8, 8, 8, 8, 8, 9, 9,10,10,10}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDqContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDqContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dq__A0__F116 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 2;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dq__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 2;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// stride_dbz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = 0
// BLOCK_DMODEL = 160
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = True
// PADDED_HEAD = False
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dq WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 160 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

