{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A @GTimeZone@ represents a time zone, at no particular point in time.
-- 
-- The @GTimeZone@ struct is refcounted and immutable.
-- 
-- Each time zone has an identifier (for example, ‘Europe\/London’) which is
-- platform dependent. See 'GI.GLib.Structs.TimeZone.timeZoneNew' for information on the
-- identifier formats. The identifier of a time zone can be retrieved using
-- 'GI.GLib.Structs.TimeZone.timeZoneGetIdentifier'.
-- 
-- A time zone contains a number of intervals. Each interval has an abbreviation
-- to describe it (for example, ‘PDT’), an offset to UTC and a flag indicating
-- if the daylight savings time is in effect during that interval. A time zone
-- always has at least one interval — interval 0. Note that interval abbreviations
-- are not the same as time zone identifiers (apart from ‘UTC’), and cannot be
-- passed to 'GI.GLib.Structs.TimeZone.timeZoneNew'.
-- 
-- Every UTC time is contained within exactly one interval, but a given
-- local time may be contained within zero, one or two intervals (due to
-- incontinuities associated with daylight savings time).
-- 
-- An interval may refer to a specific period of time (eg: the duration
-- of daylight savings time during 2010) or it may refer to many periods
-- of time that share the same properties (eg: all periods of daylight
-- savings time).  It is also possible (usually for political reasons)
-- that some properties (like the abbreviation) change between intervals
-- without other properties changing.
-- 
-- /Since: 2.26/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.TimeZone
    ( 

-- * Exported types
    TimeZone(..)                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [adjustTime]("GI.GLib.Structs.TimeZone#g:method:adjustTime"), [findInterval]("GI.GLib.Structs.TimeZone#g:method:findInterval"), [isDst]("GI.GLib.Structs.TimeZone#g:method:isDst"), [ref]("GI.GLib.Structs.TimeZone#g:method:ref"), [unref]("GI.GLib.Structs.TimeZone#g:method:unref").
-- 
-- ==== Getters
-- [getAbbreviation]("GI.GLib.Structs.TimeZone#g:method:getAbbreviation"), [getIdentifier]("GI.GLib.Structs.TimeZone#g:method:getIdentifier"), [getOffset]("GI.GLib.Structs.TimeZone#g:method:getOffset").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveTimeZoneMethod                   ,
#endif

-- ** adjustTime #method:adjustTime#

#if defined(ENABLE_OVERLOADING)
    TimeZoneAdjustTimeMethodInfo            ,
#endif
    timeZoneAdjustTime                      ,


-- ** findInterval #method:findInterval#

#if defined(ENABLE_OVERLOADING)
    TimeZoneFindIntervalMethodInfo          ,
#endif
    timeZoneFindInterval                    ,


-- ** getAbbreviation #method:getAbbreviation#

#if defined(ENABLE_OVERLOADING)
    TimeZoneGetAbbreviationMethodInfo       ,
#endif
    timeZoneGetAbbreviation                 ,


-- ** getIdentifier #method:getIdentifier#

#if defined(ENABLE_OVERLOADING)
    TimeZoneGetIdentifierMethodInfo         ,
#endif
    timeZoneGetIdentifier                   ,


-- ** getOffset #method:getOffset#

#if defined(ENABLE_OVERLOADING)
    TimeZoneGetOffsetMethodInfo             ,
#endif
    timeZoneGetOffset                       ,


-- ** isDst #method:isDst#

#if defined(ENABLE_OVERLOADING)
    TimeZoneIsDstMethodInfo                 ,
#endif
    timeZoneIsDst                           ,


-- ** new #method:new#

    timeZoneNew                             ,


-- ** newIdentifier #method:newIdentifier#

    timeZoneNewIdentifier                   ,


-- ** newLocal #method:newLocal#

    timeZoneNewLocal                        ,


-- ** newOffset #method:newOffset#

    timeZoneNewOffset                       ,


-- ** newUtc #method:newUtc#

    timeZoneNewUtc                          ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    TimeZoneRefMethodInfo                   ,
#endif
    timeZoneRef                             ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    TimeZoneUnrefMethodInfo                 ,
#endif
    timeZoneUnref                           ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.GLib.Enums as GLib.Enums

#else
import {-# SOURCE #-} qualified GI.GLib.Enums as GLib.Enums

#endif

-- | Memory-managed wrapper type.
newtype TimeZone = TimeZone (SP.ManagedPtr TimeZone)
    deriving (Eq)

instance SP.ManagedPtrNewtype TimeZone where
    toManagedPtr (TimeZone p) = p

foreign import ccall "g_time_zone_get_type" c_g_time_zone_get_type :: 
    IO GType

type instance O.ParentTypes TimeZone = '[]
instance O.HasParentTypes TimeZone

instance B.Types.TypedObject TimeZone where
    glibType = c_g_time_zone_get_type

instance B.Types.GBoxed TimeZone

-- | Convert t'TimeZone' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe TimeZone) where
    gvalueGType_ = c_g_time_zone_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr TimeZone)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr TimeZone)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed TimeZone ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList TimeZone
type instance O.AttributeList TimeZone = TimeZoneAttributeList
type TimeZoneAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method TimeZone::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "identifier"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a timezone identifier"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "TimeZone" })
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_new" g_time_zone_new :: 
    CString ->                              -- identifier : TBasicType TUTF8
    IO (Ptr TimeZone)

{-# DEPRECATED timeZoneNew ["(Since version 2.68)","Use 'GI.GLib.Structs.TimeZone.timeZoneNewIdentifier' instead, as it provides","    error reporting. Change your code to handle a potentially 'P.Nothing' return","    value."] #-}
-- | A version of 'GI.GLib.Structs.TimeZone.timeZoneNewIdentifier' which returns the UTC time zone
-- if /@identifier@/ could not be parsed or loaded.
-- 
-- If you need to check whether /@identifier@/ was loaded successfully, use
-- 'GI.GLib.Structs.TimeZone.timeZoneNewIdentifier'.
-- 
-- /Since: 2.26/
timeZoneNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@identifier@/: a timezone identifier
    -> m TimeZone
    -- ^ __Returns:__ the requested timezone
timeZoneNew identifier = liftIO $ do
    maybeIdentifier <- case identifier of
        Nothing -> return FP.nullPtr
        Just jIdentifier -> do
            jIdentifier' <- textToCString jIdentifier
            return jIdentifier'
    result <- g_time_zone_new maybeIdentifier
    checkUnexpectedReturnNULL "timeZoneNew" result
    result' <- (wrapBoxed TimeZone) result
    freeMem maybeIdentifier
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method TimeZone::new_identifier
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "identifier"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a timezone identifier"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "TimeZone" })
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_new_identifier" g_time_zone_new_identifier :: 
    CString ->                              -- identifier : TBasicType TUTF8
    IO (Ptr TimeZone)

-- | Creates a t'GI.GLib.Structs.TimeZone.TimeZone' corresponding to /@identifier@/. If /@identifier@/ cannot be
-- parsed or loaded, 'P.Nothing' is returned.
-- 
-- /@identifier@/ can either be an RFC3339\/ISO 8601 time offset or
-- something that would pass as a valid value for the @TZ@ environment
-- variable (including 'P.Nothing').
-- 
-- In Windows, /@identifier@/ can also be the unlocalized name of a time
-- zone for standard time, for example \"Pacific Standard Time\".
-- 
-- Valid RFC3339 time offsets are @\"Z\"@ (for UTC) or
-- @\"±hh:mm\"@.  ISO 8601 additionally specifies
-- @\"±hhmm\"@ and @\"±hh\"@.  Offsets are
-- time values to be added to Coordinated Universal Time (UTC) to get
-- the local time.
-- 
-- In UNIX, the @TZ@ environment variable typically corresponds
-- to the name of a file in the zoneinfo database, an absolute path to a file
-- somewhere else, or a string in
-- \"std offset [dst [offset],start[\/time],end[\/time]]\" (POSIX) format.
-- There  are  no spaces in the specification. The name of standard
-- and daylight savings time zone must be three or more alphabetic
-- characters. Offsets are time values to be added to local time to
-- get Coordinated Universal Time (UTC) and should be
-- @\"[±]hh[[:]mm[:ss]]\"@.  Dates are either
-- @\"Jn\"@ (Julian day with n between 1 and 365, leap
-- years not counted), @\"n\"@ (zero-based Julian day
-- with n between 0 and 365) or @\"Mm.w.d\"@ (day d
-- (0 \<= d \<= 6) of week w (1 \<= w \<= 5) of month m (1 \<= m \<= 12), day
-- 0 is a Sunday).  Times are in local wall clock time, the default is
-- 02:00:00.
-- 
-- In Windows, the \"tzn[+|–]hh[:mm[:ss]][dzn]\" format is used, but also
-- accepts POSIX format.  The Windows format uses US rules for all time
-- zones; daylight savings time is 60 minutes behind the standard time
-- with date and time of change taken from Pacific Standard Time.
-- Offsets are time values to be added to the local time to get
-- Coordinated Universal Time (UTC).
-- 
-- 'GI.GLib.Structs.TimeZone.timeZoneNewLocal' calls this function with the value of the
-- @TZ@ environment variable. This function itself is independent of
-- the value of @TZ@, but if /@identifier@/ is 'P.Nothing' then @\/etc\/localtime@
-- will be consulted to discover the correct time zone on UNIX and the
-- registry will be consulted or @/GetTimeZoneInformation()/@ will be used
-- to get the local time zone on Windows.
-- 
-- If intervals are not available, only time zone rules from @TZ@
-- environment variable or other means, then they will be computed
-- from year 1900 to 2037.  If the maximum year for the rules is
-- available and it is greater than 2037, then it will followed
-- instead.
-- 
-- See
-- <http://tools.ietf.org/html/rfc3339#section-5.6 RFC3339 §5.6>
-- for a precise definition of valid RFC3339 time offsets
-- (the @time-offset@ expansion) and ISO 8601 for the
-- full list of valid time offsets.  See
-- <http://www.gnu.org/s/libc/manual/html_node/TZ-Variable.html The GNU C Library manual>
-- for an explanation of the possible
-- values of the @TZ@ environment variable. See
-- <http://msdn.microsoft.com/en-us/library/ms912391%28v=winembedded.11%29.aspx Microsoft Time Zone Index Values>
-- for the list of time zones on Windows.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.TimeZone.timeZoneUnref'
-- when you are done with it.
-- 
-- /Since: 2.68/
timeZoneNewIdentifier ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@identifier@/: a timezone identifier
    -> m (Maybe TimeZone)
    -- ^ __Returns:__ the requested timezone, or 'P.Nothing' on
    --     failure
timeZoneNewIdentifier identifier = liftIO $ do
    maybeIdentifier <- case identifier of
        Nothing -> return FP.nullPtr
        Just jIdentifier -> do
            jIdentifier' <- textToCString jIdentifier
            return jIdentifier'
    result <- g_time_zone_new_identifier maybeIdentifier
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed TimeZone) result'
        return result''
    freeMem maybeIdentifier
    return maybeResult

#if defined(ENABLE_OVERLOADING)
#endif

-- method TimeZone::new_local
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "TimeZone" })
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_new_local" g_time_zone_new_local :: 
    IO (Ptr TimeZone)

-- | Creates a t'GI.GLib.Structs.TimeZone.TimeZone' corresponding to local time.  The local time
-- zone may change between invocations to this function; for example,
-- if the system administrator changes it.
-- 
-- This is equivalent to calling 'GI.GLib.Structs.TimeZone.timeZoneNew' with the value of
-- the @TZ@ environment variable (including the possibility of 'P.Nothing').
-- 
-- You should release the return value by calling 'GI.GLib.Structs.TimeZone.timeZoneUnref'
-- when you are done with it.
-- 
-- /Since: 2.26/
timeZoneNewLocal ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m TimeZone
    -- ^ __Returns:__ the local timezone
timeZoneNewLocal  = liftIO $ do
    result <- g_time_zone_new_local
    checkUnexpectedReturnNULL "timeZoneNewLocal" result
    result' <- (wrapBoxed TimeZone) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method TimeZone::new_offset
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "seconds"
--           , argType = TBasicType TInt32
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "offset to UTC, in seconds"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "TimeZone" })
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_new_offset" g_time_zone_new_offset :: 
    Int32 ->                                -- seconds : TBasicType TInt32
    IO (Ptr TimeZone)

-- | Creates a t'GI.GLib.Structs.TimeZone.TimeZone' corresponding to the given constant offset from UTC,
-- in seconds.
-- 
-- This is equivalent to calling 'GI.GLib.Structs.TimeZone.timeZoneNew' with a string in the form
-- @[+|-]hh[:mm[:ss]]@.
-- 
-- It is possible for this function to fail if /@seconds@/ is too big (greater than
-- 24 hours), in which case this function will return the UTC timezone for
-- backwards compatibility. To detect failures like this, use
-- 'GI.GLib.Structs.TimeZone.timeZoneNewIdentifier' directly.
-- 
-- /Since: 2.58/
timeZoneNewOffset ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Int32
    -- ^ /@seconds@/: offset to UTC, in seconds
    -> m TimeZone
    -- ^ __Returns:__ a timezone at the given offset from UTC, or UTC on
    --   failure
timeZoneNewOffset seconds = liftIO $ do
    result <- g_time_zone_new_offset seconds
    checkUnexpectedReturnNULL "timeZoneNewOffset" result
    result' <- (wrapBoxed TimeZone) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method TimeZone::new_utc
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "TimeZone" })
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_new_utc" g_time_zone_new_utc :: 
    IO (Ptr TimeZone)

-- | Creates a t'GI.GLib.Structs.TimeZone.TimeZone' corresponding to UTC.
-- 
-- This is equivalent to calling 'GI.GLib.Structs.TimeZone.timeZoneNew' with a value like
-- \"Z\", \"UTC\", \"+00\", etc.
-- 
-- You should release the return value by calling 'GI.GLib.Structs.TimeZone.timeZoneUnref'
-- when you are done with it.
-- 
-- /Since: 2.26/
timeZoneNewUtc ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m TimeZone
    -- ^ __Returns:__ the universal timezone
timeZoneNewUtc  = liftIO $ do
    result <- g_time_zone_new_utc
    checkUnexpectedReturnNULL "timeZoneNewUtc" result
    result' <- (wrapBoxed TimeZone) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method TimeZone::adjust_time
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GTimeType of @time_"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "time_"
--           , argType = TBasicType TInt64
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a pointer to a number of seconds since January 1, 1970"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_adjust_time" g_time_zone_adjust_time :: 
    Ptr TimeZone ->                         -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    CUInt ->                                -- type : TInterface (Name {namespace = "GLib", name = "TimeType"})
    Ptr Int64 ->                            -- time_ : TBasicType TInt64
    IO Int32

-- | Finds an interval within /@tz@/ that corresponds to the given /@time_@/,
-- possibly adjusting /@time_@/ if required to fit into an interval.
-- The meaning of /@time_@/ depends on /@type@/.
-- 
-- This function is similar to 'GI.GLib.Structs.TimeZone.timeZoneFindInterval', with the
-- difference that it always succeeds (by making the adjustments
-- described below).
-- 
-- In any of the cases where 'GI.GLib.Structs.TimeZone.timeZoneFindInterval' succeeds then
-- this function returns the same value, without modifying /@time_@/.
-- 
-- This function may, however, modify /@time_@/ in order to deal with
-- non-existent times.  If the non-existent local /@time_@/ of 02:30 were
-- requested on March 14th 2010 in Toronto then this function would
-- adjust /@time_@/ to be 03:00 and return the interval containing the
-- adjusted time.
-- 
-- /Since: 2.26/
timeZoneAdjustTime ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> GLib.Enums.TimeType
    -- ^ /@type@/: the t'GI.GLib.Enums.TimeType' of /@time_@/
    -> Int64
    -- ^ /@time_@/: a pointer to a number of seconds since January 1, 1970
    -> m ((Int32, Int64))
    -- ^ __Returns:__ the interval containing /@time_@/, never -1
timeZoneAdjustTime tz type_ time_ = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    let type_' = (fromIntegral . fromEnum) type_
    time_' <- allocMem :: IO (Ptr Int64)
    poke time_' time_
    result <- g_time_zone_adjust_time tz' type_' time_'
    time_'' <- peek time_'
    touchManagedPtr tz
    freeMem time_'
    return (result, time_'')

#if defined(ENABLE_OVERLOADING)
data TimeZoneAdjustTimeMethodInfo
instance (signature ~ (GLib.Enums.TimeType -> Int64 -> m ((Int32, Int64))), MonadIO m) => O.OverloadedMethod TimeZoneAdjustTimeMethodInfo TimeZone signature where
    overloadedMethod = timeZoneAdjustTime

instance O.OverloadedMethodInfo TimeZoneAdjustTimeMethodInfo TimeZone where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.TimeZone.timeZoneAdjustTime",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-TimeZone.html#v:timeZoneAdjustTime"
        })


#endif

-- method TimeZone::find_interval
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GTimeType of @time_"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "time_"
--           , argType = TBasicType TInt64
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a number of seconds since January 1, 1970"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_find_interval" g_time_zone_find_interval :: 
    Ptr TimeZone ->                         -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    CUInt ->                                -- type : TInterface (Name {namespace = "GLib", name = "TimeType"})
    Int64 ->                                -- time_ : TBasicType TInt64
    IO Int32

-- | Finds an interval within /@tz@/ that corresponds to the given /@time_@/.
-- The meaning of /@time_@/ depends on /@type@/.
-- 
-- If /@type@/ is 'GI.GLib.Enums.TimeTypeUniversal' then this function will always
-- succeed (since universal time is monotonic and continuous).
-- 
-- Otherwise /@time_@/ is treated as local time.  The distinction between
-- 'GI.GLib.Enums.TimeTypeStandard' and 'GI.GLib.Enums.TimeTypeDaylight' is ignored except in
-- the case that the given /@time_@/ is ambiguous.  In Toronto, for example,
-- 01:30 on November 7th 2010 occurred twice (once inside of daylight
-- savings time and the next, an hour later, outside of daylight savings
-- time).  In this case, the different value of /@type@/ would result in a
-- different interval being returned.
-- 
-- It is still possible for this function to fail.  In Toronto, for
-- example, 02:00 on March 14th 2010 does not exist (due to the leap
-- forward to begin daylight savings time).  -1 is returned in that
-- case.
-- 
-- /Since: 2.26/
timeZoneFindInterval ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> GLib.Enums.TimeType
    -- ^ /@type@/: the t'GI.GLib.Enums.TimeType' of /@time_@/
    -> Int64
    -- ^ /@time_@/: a number of seconds since January 1, 1970
    -> m Int32
    -- ^ __Returns:__ the interval containing /@time_@/, or -1 in case of failure
timeZoneFindInterval tz type_ time_ = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    let type_' = (fromIntegral . fromEnum) type_
    result <- g_time_zone_find_interval tz' type_' time_
    touchManagedPtr tz
    return result

#if defined(ENABLE_OVERLOADING)
data TimeZoneFindIntervalMethodInfo
instance (signature ~ (GLib.Enums.TimeType -> Int64 -> m Int32), MonadIO m) => O.OverloadedMethod TimeZoneFindIntervalMethodInfo TimeZone signature where
    overloadedMethod = timeZoneFindInterval

instance O.OverloadedMethodInfo TimeZoneFindIntervalMethodInfo TimeZone where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.TimeZone.timeZoneFindInterval",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-TimeZone.html#v:timeZoneFindInterval"
        })


#endif

-- method TimeZone::get_abbreviation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "interval"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an interval within the timezone"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_get_abbreviation" g_time_zone_get_abbreviation :: 
    Ptr TimeZone ->                         -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    Int32 ->                                -- interval : TBasicType TInt
    IO CString

-- | Determines the time zone abbreviation to be used during a particular
-- /@interval@/ of time in the time zone /@tz@/.
-- 
-- For example, in Toronto this is currently \"EST\" during the winter
-- months and \"EDT\" during the summer months when daylight savings time
-- is in effect.
-- 
-- /Since: 2.26/
timeZoneGetAbbreviation ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> Int32
    -- ^ /@interval@/: an interval within the timezone
    -> m T.Text
    -- ^ __Returns:__ the time zone abbreviation, which belongs to /@tz@/
timeZoneGetAbbreviation tz interval = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    result <- g_time_zone_get_abbreviation tz' interval
    checkUnexpectedReturnNULL "timeZoneGetAbbreviation" result
    result' <- cstringToText result
    touchManagedPtr tz
    return result'

#if defined(ENABLE_OVERLOADING)
data TimeZoneGetAbbreviationMethodInfo
instance (signature ~ (Int32 -> m T.Text), MonadIO m) => O.OverloadedMethod TimeZoneGetAbbreviationMethodInfo TimeZone signature where
    overloadedMethod = timeZoneGetAbbreviation

instance O.OverloadedMethodInfo TimeZoneGetAbbreviationMethodInfo TimeZone where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.TimeZone.timeZoneGetAbbreviation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-TimeZone.html#v:timeZoneGetAbbreviation"
        })


#endif

-- method TimeZone::get_identifier
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_get_identifier" g_time_zone_get_identifier :: 
    Ptr TimeZone ->                         -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    IO CString

-- | Get the identifier of this t'GI.GLib.Structs.TimeZone.TimeZone', as passed to 'GI.GLib.Structs.TimeZone.timeZoneNew'.
-- If the identifier passed at construction time was not recognised, @UTC@ will
-- be returned. If it was 'P.Nothing', the identifier of the local timezone at
-- construction time will be returned.
-- 
-- The identifier will be returned in the same format as provided at
-- construction time: if provided as a time offset, that will be returned by
-- this function.
-- 
-- /Since: 2.58/
timeZoneGetIdentifier ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> m T.Text
    -- ^ __Returns:__ identifier for this timezone
timeZoneGetIdentifier tz = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    result <- g_time_zone_get_identifier tz'
    checkUnexpectedReturnNULL "timeZoneGetIdentifier" result
    result' <- cstringToText result
    touchManagedPtr tz
    return result'

#if defined(ENABLE_OVERLOADING)
data TimeZoneGetIdentifierMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod TimeZoneGetIdentifierMethodInfo TimeZone signature where
    overloadedMethod = timeZoneGetIdentifier

instance O.OverloadedMethodInfo TimeZoneGetIdentifierMethodInfo TimeZone where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.TimeZone.timeZoneGetIdentifier",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-TimeZone.html#v:timeZoneGetIdentifier"
        })


#endif

-- method TimeZone::get_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "interval"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an interval within the timezone"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt32)
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_get_offset" g_time_zone_get_offset :: 
    Ptr TimeZone ->                         -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    Int32 ->                                -- interval : TBasicType TInt
    IO Int32

-- | Determines the offset to UTC in effect during a particular /@interval@/
-- of time in the time zone /@tz@/.
-- 
-- The offset is the number of seconds that you add to UTC time to
-- arrive at local time for /@tz@/ (ie: negative numbers for time zones
-- west of GMT, positive numbers for east).
-- 
-- /Since: 2.26/
timeZoneGetOffset ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> Int32
    -- ^ /@interval@/: an interval within the timezone
    -> m Int32
    -- ^ __Returns:__ the number of seconds that should be added to UTC to get the
    --          local time in /@tz@/
timeZoneGetOffset tz interval = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    result <- g_time_zone_get_offset tz' interval
    touchManagedPtr tz
    return result

#if defined(ENABLE_OVERLOADING)
data TimeZoneGetOffsetMethodInfo
instance (signature ~ (Int32 -> m Int32), MonadIO m) => O.OverloadedMethod TimeZoneGetOffsetMethodInfo TimeZone signature where
    overloadedMethod = timeZoneGetOffset

instance O.OverloadedMethodInfo TimeZoneGetOffsetMethodInfo TimeZone where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.TimeZone.timeZoneGetOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-TimeZone.html#v:timeZoneGetOffset"
        })


#endif

-- method TimeZone::is_dst
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "interval"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an interval within the timezone"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_is_dst" g_time_zone_is_dst :: 
    Ptr TimeZone ->                         -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    Int32 ->                                -- interval : TBasicType TInt
    IO CInt

-- | Determines if daylight savings time is in effect during a particular
-- /@interval@/ of time in the time zone /@tz@/.
-- 
-- /Since: 2.26/
timeZoneIsDst ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> Int32
    -- ^ /@interval@/: an interval within the timezone
    -> m Bool
    -- ^ __Returns:__ 'P.True' if daylight savings time is in effect
timeZoneIsDst tz interval = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    result <- g_time_zone_is_dst tz' interval
    let result' = (/= 0) result
    touchManagedPtr tz
    return result'

#if defined(ENABLE_OVERLOADING)
data TimeZoneIsDstMethodInfo
instance (signature ~ (Int32 -> m Bool), MonadIO m) => O.OverloadedMethod TimeZoneIsDstMethodInfo TimeZone signature where
    overloadedMethod = timeZoneIsDst

instance O.OverloadedMethodInfo TimeZoneIsDstMethodInfo TimeZone where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.TimeZone.timeZoneIsDst",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-TimeZone.html#v:timeZoneIsDst"
        })


#endif

-- method TimeZone::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "TimeZone" })
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_ref" g_time_zone_ref :: 
    Ptr TimeZone ->                         -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    IO (Ptr TimeZone)

-- | Increases the reference count on /@tz@/.
-- 
-- /Since: 2.26/
timeZoneRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> m TimeZone
    -- ^ __Returns:__ a new reference to /@tz@/.
timeZoneRef tz = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    result <- g_time_zone_ref tz'
    checkUnexpectedReturnNULL "timeZoneRef" result
    result' <- (wrapBoxed TimeZone) result
    touchManagedPtr tz
    return result'

#if defined(ENABLE_OVERLOADING)
data TimeZoneRefMethodInfo
instance (signature ~ (m TimeZone), MonadIO m) => O.OverloadedMethod TimeZoneRefMethodInfo TimeZone signature where
    overloadedMethod = timeZoneRef

instance O.OverloadedMethodInfo TimeZoneRefMethodInfo TimeZone where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.TimeZone.timeZoneRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-TimeZone.html#v:timeZoneRef"
        })


#endif

-- method TimeZone::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "tz"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TimeZone" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTimeZone" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_time_zone_unref" g_time_zone_unref :: 
    Ptr TimeZone ->                         -- tz : TInterface (Name {namespace = "GLib", name = "TimeZone"})
    IO ()

-- | Decreases the reference count on /@tz@/.
-- 
-- /Since: 2.26/
timeZoneUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TimeZone
    -- ^ /@tz@/: a t'GI.GLib.Structs.TimeZone.TimeZone'
    -> m ()
timeZoneUnref tz = liftIO $ do
    tz' <- unsafeManagedPtrGetPtr tz
    g_time_zone_unref tz'
    touchManagedPtr tz
    return ()

#if defined(ENABLE_OVERLOADING)
data TimeZoneUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod TimeZoneUnrefMethodInfo TimeZone signature where
    overloadedMethod = timeZoneUnref

instance O.OverloadedMethodInfo TimeZoneUnrefMethodInfo TimeZone where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.TimeZone.timeZoneUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-TimeZone.html#v:timeZoneUnref"
        })


#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveTimeZoneMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveTimeZoneMethod "adjustTime" o = TimeZoneAdjustTimeMethodInfo
    ResolveTimeZoneMethod "findInterval" o = TimeZoneFindIntervalMethodInfo
    ResolveTimeZoneMethod "isDst" o = TimeZoneIsDstMethodInfo
    ResolveTimeZoneMethod "ref" o = TimeZoneRefMethodInfo
    ResolveTimeZoneMethod "unref" o = TimeZoneUnrefMethodInfo
    ResolveTimeZoneMethod "getAbbreviation" o = TimeZoneGetAbbreviationMethodInfo
    ResolveTimeZoneMethod "getIdentifier" o = TimeZoneGetIdentifierMethodInfo
    ResolveTimeZoneMethod "getOffset" o = TimeZoneGetOffsetMethodInfo
    ResolveTimeZoneMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveTimeZoneMethod t TimeZone, O.OverloadedMethod info TimeZone p) => OL.IsLabel t (TimeZone -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveTimeZoneMethod t TimeZone, O.OverloadedMethod info TimeZone p, R.HasField t TimeZone p) => R.HasField t TimeZone p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveTimeZoneMethod t TimeZone, O.OverloadedMethodInfo info TimeZone) => OL.IsLabel t (O.MethodProxy info TimeZone) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


